'use strict'

import debug from '../utils/debug'
import base from './_base'
import slugalize from './slugalize'
import serialize from './serialize'
import map_styles from '../explore/map/styles'
import url from '../utils/url'

import Queue from '../utils/queue'

window.$ = window.jQuery

/*
 * text
 *
 */
export class text extends base {

    init() {

        let time = null

        $('input', this.$).on('input', () => {
            clearTimeout( time )
            time = setTimeout(() => {
                this.collect()
            }, 100 )
		});

    }

    get_label() {
        return $('input', this.$ ).val();
    }

}

/*
 * hidden
 *
 */
export class hidden extends base {

    init() {

        $('input', this.$).on('input', () => {
			this.collect()
		})

    }

}

/*
 * textarea
 *
 */
export class textarea extends base {

    init() {

        let time = null

        $('textarea', this.$).on('input', () => {
            clearTimeout( time )
            time = setTimeout(() => {
                this.collect()
            }, 100 )
		})

    }

}

/*
 * checkbox
 *
 */
export class checkbox extends base {

    init() {

        $('[type="checkbox"]', this.$).on('change', (e) => {
			$('[type="hidden"]', this.$).val( e.target.checked ? '1' : '' )
                this.collect()
		});

    }

    get_label() {
        return $('input', this.$).is(':checked') ? $('.ulz--collect', this.$).html() : null;
    }

}

/*
 * toggle
 *
 */
export class toggle extends base {

    init() {

        $('[type="checkbox"]', this.$).on('change', (e) => {
			$('[type="hidden"]', this.$).val( e.target.checked ? '1' : '' )
            this.collect()
		});

    }

}

/*
 * checklist
 *
 */
export class checklist extends base {

    init() {

		$('[type="checkbox"]', this.$).on('change', () => {
			$('[type="hidden"]', this.$).prop('disabled', $('[type="checkbox"]', this.$).serialize() )
            this.collect()
		});

    }

    get_label() {
        let selected = []
        $('input[type="checkbox"]:checked', this.$).each(( index, e ) => {
            selected.push( $(e).closest('.ulz-checkbox').find('.ulz--collect').html() )
        })
        return selected.join(', ')
    }

}

/*
 * options
 *
 */
export class options extends base {

    init() {

        this.$input = $('textarea', this.$)
        this.$items = $('.ulz-opts-items', this.$)

        $('.ulz-button-opts-add', this.$).on('click', () => this.add())
		$('.ulz-button-opts-save', this.$).on('click', () => this.save())

    }

    add() {

        this.$.addClass('ulz-expand')
        this.$input.val( this.$input.val().split('%%').join('\n') )

    }

    save() {

        this.$items.html('')

        let lines = this.$input.val().split('\n').join('%%').split('%%')

        for( let k = 0; k < lines.length; k++ ) {

            let option = lines[ k ].replace(/([ \t]+)?:([ \t]+)?/g, ':').split(':')

            if( option[0].trim() === '' ) { continue }
            if( ! option[1] ) { option[1] = option[0] }

            this.$items.append('<li><input type="text" value="' + option[1] + '" disabled></li>')

        }

        this.$input.val( lines.join('%%') )

        this.$.removeClass('ulz-expand')

        this.collect()

    }

}

/*
 * number
 *
 */
export class number extends base {

    init() {

        let type = this.$.data('input-type');

        if( type == 'range' ) {
            this.range();
        }else if( type == 'stepper' ) {
            this.stepper();
        }else{
            // number
        }

        let time = null

        $('input', this.$).on('input', () => {
            clearTimeout( time )
            time = setTimeout(() => {
                this.collect()
            }, 100 )
		});

    }

    range() {

        let input = $('input', this.$);

        input.on('input', (e) => {

            $('.ulz-number-range-text', this.$).html( input.data('format').replace( /%s/g, e.target.value ) );

        });

    }

    stepper() {

        let input = $('[type="number"]', this.$);
        let step = input.data('step') !== 'undefined' ? input.data('step') : 1;

        this.$.on('click', '.ulz-stepper-button', (e) => {

            $(e.currentTarget).data('action') == 'increase' ? input.get(0).stepUp( step ) : input.get(0).stepDown( step )
            $('.ulz-stepper-text', this.$).html( input.data('format').replace( /%s/g, input.val() ) )
            input.trigger('input')

        });

    }

}

/*
 * repeater
 *
 */
export class repeater extends base {

    init() {

        let self = this;

        this.$repeater = $('>.ulz-repeater', this.$);
        this.$select = $('>.ulz-repeater-select', this.$repeater);
        this.$items = $('>.ulz-repeater-items', this.$repeater);

        this.items();
		this.sortable();

		$('.ulz-button', this.$select).on('click', () => this.additem());

    }

	items() {

        $('>.ulz-repeater-item', this.$items).not('.ulz-item-ready').each((index, element) => {
			new repeater_item( $(element) );
        });

    }

	sortable() {

        let _this = this

		$( '>.ulz-repeater-items', this.$repeater ).sortable({
			distance: 5,
			placeholder: 'ulz-item-dummy',
			handle: '.ulz-item-label',
			start: function( e, ui ) {
                ui.placeholder.attr('data-id', ui.item.data('id'))
				ui.placeholder.height( ui.item.height() )
			},
			update: function( e, ui ) {
				_this.collect()
			}
		});

	}

	additem() {

        let template = $('select', this.$select).val()

        if( ! template.length ) {
            alert( window.utillz_core_vars.localize.select_module );
            return;
        }

		this.ajaxing()

		let schema = JSON.parse( $('>.ulz-repeater-schema', this.$repeater).val() )

        let data = {
            action: 'utillz-field-repeater',
            post_id: window.utillz_core_vars.post_id,
            schema: JSON.stringify( schema[ template ] ),
            template: template,
            storage: this.$.attr('data-storage')
        }

		$.ajax({
			type: 'POST',
			url: window.utillz_core_vars.admin_ajax,
			data: data,
			success: ( response ) => {

				this.$.removeClass('ulz-ajaxing')

				if( response.success ) {

					let item_element = $( response.html )
					$('>.ulz-repeater-items', this.$repeater).append( item_element )

                    $(document).trigger('utillz:form/init')

                    new repeater_item( item_element ).expand()

                    this.collect()

				}

			},
		});

	}

}

/*
 * repeater item
 *
 */
export class repeater_item {

    constructor( item ) {

        this.$ = item;
        this.$row = $('>.ulz-item-row', this.$);

        $('.ulz-item-remove', this.$row).on('click', (e) => this.remove(e));
        $('.ulz-item-clone', this.$row).on('click', (e) => this.clone(e));
        $('.ulz-item-hide', this.$row).on('click', (e) => this.hide(e));
		$('.ulz-item-label', this.$row).on('click', (e) => this.expand(e));
		$( '>.ulz-repeater-content > [data-id="' + this.$.data('heading') + '"] input', this.$ ).on('input', (e) => this.heading(e));

        item.addClass('ulz-item-ready');

    }

    collect() {
        $(document).trigger('utillz:form/changed');
    }

	remove(e) {
		$(e.currentTarget).closest('.ulz-repeater-item').remove();
		this.collect()
	}

	clone(e) {

        let $e = $(e.currentTarget)
        let $item = $e.closest('.ulz-repeater-item')
        let $clone = $item.clone()

        $clone.find('.ulz-field-ready').removeClass('ulz-field-ready')
        $clone.insertAfter( $item )

        $('.ulz-field', $clone).each((index, e) => {

            let $field = $(e)
            let type = $field.attr('data-type')

            // re-populate auto-keys
            if( type == 'auto-key' ) {
                new auto_key( $field ).repopulate()
            }

        })

        $(document).trigger('utillz:form/init')
        new repeater_item( $clone ).unexpand()
		this.collect()

	}

    hide(e) {

        let $e = $(e.currentTarget)
        let $item = $e.closest('.ulz-repeater-item')
        let $check = $item.find(`[name='_item_hidden']:first`)
        let is_checked = $check.val().length

        $check.val( is_checked ? '' : '1' ).trigger('input')
        $item[ ! is_checked ? 'addClass' : 'removeClass' ]('ulz-item-hidden')

        // $item.toggleClass('ulz-item-hidden')

    }

	unexpand() {

		this.$.removeClass('ulz-expand')

	}

	expand() {

        if( this.$.hasClass('ulz-item-empty') ) {
            return;
        }

		this.$.parent().children().not( this.$ ).removeClass('ulz-expand')
		this.$.toggleClass('ulz-expand')

	}

	heading(e) {

		$('>.ulz-item-row .ulz-item-title', this.$).html( $( e.currentTarget ).val().replace(/(<([^>]+)>)/gi, "") )

	}

}

/*
 * radio
 *
 */
export class radio extends base {

    init() {

        $('[type="radio"]', this.$).on('change', () => {
			this.collect()
		});

    }

    get_label() {
        return $('input:checked', this.$).length ? $('input:checked', this.$).closest('.ulz-radio').find('span').html() : null;
    }
}

/*
 * radio image
 *
 */
export class radio_image extends base {

    init() {

        $('[type="radio"]', this.$).on('change', () => {
			this.collect()
		});

    }
}

/*
 * select
 *
 */
export class select extends base {

    init() {

        $('select', this.$).on('change', () => {
            this.collect()
        })

        // fix for empty inputs in post
		$('select[multiple]', this.$).on('change', (e) => {
			$( '>[type="hidden"]', this.$ ).prop('disabled', $( 'select', this.$ ).serialize() );
		})

    }

    get_label() {
        return $('select', this.$).val() ? $('option:selected', this.$).html() : null;
    }

}

/*
 * icon
 *
 */
export class icon extends base {

    init() {

        try {
            this.icon_data = JSON.parse( $('.ulz--input', this.$).val() )
            if( Array.isArray( this.icon_data ) ) {
                this.icon_data = this.icon_data[0]
            }
        }catch(e) {
            this.icon_data = {
                'set': '',
                'icon': '',
            }
        }

        $(`[data-action='icon']`, this.$).on('click', e => this.toggle(e))
        $(`[data-action='cancel']`, this.$).on('click', e => this.cancel(e))

    }

    toggle() {

        window.Utillz_Core.modal.open('field-icon', {
            set: this.icon_data.set,
            icon: this.icon_data.icon,
        })

        $(document).off('utillz:icon/push').on('utillz:icon/push', (e, params) => {

            let icon_data = {
                'set': params.set,
                'icon': params.icon
            }

            $('.ulz--input', this.$).val( JSON.stringify( [ icon_data ] ) )
            $('.ulz--input-set', this.$).val( params.set )
            $('.ulz--input-icon', this.$).val( params.icon )
            $('.ulz--preview', this.$).html( params.preview )

            this.collect()

        })

    }

    cancel() {

        $('.ulz--input', this.$).val('')
        $('.ulz--input-set', this.$).val('')
        $('.ulz--input-icon', this.$).val('')
        $('.ulz--preview', this.$).html('')

        this.collect()

    }

}

/*
 * key
 *
 */
export class key extends base {

    init() {

        this.hidden = $('[type="hidden"]', this.$)
		this.select = $('select', this.$)
		this.buttons = $('.ulz-button', this.$)
		this.text = $('[type="text"]', this.$)

        this.buttons.on('click', () => this.on_click())
        this.text.on('focusout', (e) => this.on_focusout(e))
		this.select.on('change', (e) => this.on_change(e))

    }

    on_click() {

        this.$.toggleClass('ulz-is-defined')
        let input = this.$.hasClass('ulz-is-defined') ? this.select : this.text

        this.hidden.val( input.val() )

    }

    on_focusout(e) {

        let input = $( e.currentTarget )
        let slugalized = slugalize( input.val() )

        input.val( slugalized )
        this.hidden.val( slugalized )

        this.collect()

    }

    on_change(e) {

        let input = $( e.currentTarget )
        this.hidden.val( input.val() )

        this.collect()

    }

}

/*
 * auto key
 *
 */
export class auto_key extends base {

    init() {

        this.$input = $('input', this.$)

        this.$input.on('input', () => {
            this.populate()
			this.collect()
		})

        this.populate()

    }

    repopulate() {
        this.$input.val( this.generate( parseInt( this.$input.attr('data-char-length') ), this.$input.attr('data-char-type') ) )
    }

    populate() {

        if( ! this.$input.val() ) {
            this.repopulate()
        }

    }

    generate( char_length, char_type ) {

        char_length = char_length || 30
        char_type = char_type || 'any'

        let result = []
        let characters

        switch( characters ) {
            case 'string': characters = 'abcdefghijklmnopqrstuvwxyz'
                break;
            case 'numeric': characters = '0123456789'
                break;
            default:  characters = 'abcdefghijklmnopqrstuvwxyz0123456789'
        }

        for( var i = 0; i < char_length; i++ ) {
            result.push( characters.charAt( Math.floor( Math.random() * characters.length ) ) )
        }

        return result.join('')

    }

}

/*
 * map
 *
 */
export class map extends base {

    init() {

        this.map = $('.ulz-map', this.$)
        this.lat = $('.ulz-map-lat', this.$)
        this.lng = $('.ulz-map-lng', this.$)
        this.address = $('.ulz-map-address', this.$)

        this.country = $('.ulz-map-country', this.$)
        this.city = $('.ulz-map-city', this.$)
        this.city_alt = $('.ulz-map-city-alt', this.$)

        this.set()

    }

    set() {

        if( typeof window.google == 'undefined' ) {
            $( 'span', this.$ ).removeClass('ulz-none')
            return;
        }

		// location
		let myLatLng = {
			lat: parseFloat( this.lat.val() ),
			lng: parseFloat( this.lng.val() )
		};

		// init map
		let map = new google.maps.Map( this.map.get(0), {
			center: myLatLng,
			zoom: 15,
			mapTypeControl: false,
			streetViewControl: false,
			scrollwheel: false,
			styles: map_styles()
		});

		// set marker
		let marker = new google.maps.Marker({
			position: myLatLng,
			map: map,
			title: 'Marker',
			draggable: true,
		});

		let geocoder = new google.maps.Geocoder()

		// marker drag listener
		google.maps.event.addListener( marker, 'dragend', ( marker ) => {


            let geo_args = {
                'latLng': marker.latLng
            }

			geocoder.geocode( geo_args, ( results, status ) => {

				if( status == google.maps.GeocoderStatus.OK ) {
					if ( results[0] ) {

                        let geo = this.extract_geo( results )

						this.set_values(
							results[0].formatted_address,
							marker.latLng.lat(),
							marker.latLng.lng(),
                            geo
						);

					}
				}else{

                    alert('Geocode was not successful for the following reason: ' + status)

                }
			})


		});

        let geo_args = []

        if( window.utillz_core_vars.map.geolocation_restrictions.length ) {
            geo_args.componentRestrictions = {
                country: window.utillz_core_vars.map.geolocation_restrictions,
            }
        }

		// create the search box and link it to the UI element.
		let searchBox = new google.maps.places.Autocomplete( this.address.get(0), geo_args );

		// bias the SearchBox results towards current map's viewport.
		map.addListener( 'bounds_changed', () => {
			searchBox.setBounds( map.getBounds() )
		})

		// Listen for the event fired when the user selects a prediction and retrieve
		// more details for that place.
		searchBox.addListener( 'place_changed', () => {

			let place = searchBox.getPlace();

			if ( ! place.geometry || ! place.geometry.location ) {
				return;
			}

			// For each place, get the icon, name and location.
			var bounds = new google.maps.LatLngBounds();

			if( typeof place !== 'undefined' ) {

				if ( ! place.geometry ) {
					console.log('Returned place contains no geometry');
					return;
				}

				if ( place.geometry.viewport ) { // Only geocodes have viewport.
					bounds.union( place.geometry.viewport )
				} else {
					bounds.extend( place.geometry.location )
				}

				marker.setPosition( place.geometry.location )
				map.fitBounds( bounds )

                let geo_args = {
                    'latLng': place.geometry.location
                }

                // get
                geocoder.geocode( geo_args, ( results, status ) => {

    				if ( status == google.maps.GeocoderStatus.OK ) {
    					if ( results[0] ) {

                            let geo = this.extract_geo( results )

                            this.set_values(
            					place.formatted_address,
            					place.geometry.location.lat(),
            					place.geometry.location.lng(),
                                geo
            				);

    					}
                    }else{

                        alert('Geocode was not successful for the following reason: ' + status)

                    }
    			})

			}

		});

		// TODO: add get geolocation icon button in searchBox input field

    }

    extract_geo( results ) {

        let country = null,
            city = null,
            city_alt = null

        for( let i = 0; i < results.length; i++ ) {

            if( results[i].types.includes('locality') ) {
                city = results[i].formatted_address
                continue;
            }

            if( results[i].types.includes('administrative_area_level_1') ) {
                city_alt = results[i].formatted_address
                continue;
            }

            if( results[i].types.includes('country') ) {
                country = results[i].formatted_address
                continue;
            }

        }

        return {
            country: country,
            city: city,
            city_alt: city_alt
        };

    }

    set_values( _address, _lat, _lng, geo ) {

        geo = geo || null

		this.address.val( _address );

		let lat;
		lat = this.lat;
		lat = lat.add( lat.prev() );
		lat.val( _lat );

		let lng;
		lng = this.lng;
		lng = lng.add( lng.prev() );
		lng.val( _lng );

        if( geo ) {
            this.country.val( geo.country )
            this.city.val( geo.city )
            this.city_alt.val( geo.city_alt )
        }

		this.collect()

	}

}

/*
 * upload
 *
 */
export class upload extends base {

    init_admin() {

        this.is_multiple = this.$.data('multiple');
        this.upload_type = this.$.data('upload-type');
        this.$input = $('.ulz-upload-input', this.$);
        this.$preview = $('.ulz-image-preview', this.$);

        if( this.is_multiple ) {
            this.reorder();
        }

        this.$.on('click', '.ulz-image-remove', (e) => this.remove(e));

        $('.ulz-upload-button', this.$).on('click', (e) => {

            e.preventDefault();

            let type = []
            if( this.upload_type == 'image' ) {
                type = [
                    'image',
                ]
            }

            const ulz_media_uploader = wp.media.frames.file_frame = wp.media({ // extend the wp.media object
                library: {
                    type: type,
                },
                multiple: this.is_multiple,
            });

            ulz_media_uploader.on('select', () => { // when a file is selected, grab the url and set it as the text field's value

                let attachment_output = '';

                // single
                if( ! this.is_multiple ) {

                    let attachment = ulz_media_uploader.state().get('selection').first().toJSON();

                    let attachment_url = null;
                    if( typeof attachment.sizes == 'undefined' ) {
                        // no image size support
                    }else if( typeof attachment.sizes.ulz_thumbnail !== 'undefined' ) {
                        attachment_url = attachment.sizes.ulz_thumbnail.url
                    }else if( typeof attachment.sizes.thumbnail !== 'undefined' ) {
                        attachment_url = attachment.sizes.thumbnail.url
                    }else{
                        attachment_url = attachment.sizes.full.url
                    }

                    let attachment_output = attachment.url;

                    // set input value
                    this.$input
                        .val( JSON.stringify([{
                            id: attachment.id,
                        }]))
                        .trigger('input');

                    // preview
                    this.preview( attachment_url, attachment.id );

                    this.collect()

                }
                // multiple
                else{

                    let _ids = [];

                    try {
                        let o = JSON.parse( this.$input.val() );
                        if ( o && typeof o === 'object' ) {
                            if( this.$input.val() !== '' ) {
                                _ids = JSON.parse( this.$input.val() );
                            }
                        }
                    } catch (e) {}

                    let selection = ulz_media_uploader.state().get( 'selection' );
                    selection.map(( attachment ) => {

                        attachment = attachment.toJSON();

                        let attachment_url = null;
                        if( typeof attachment.sizes == 'undefined' ) {
                            // no image size support
                        }else if( typeof attachment.sizes.ulz_thumbnail !== 'undefined' ) {
                            attachment_url = attachment.sizes.ulz_thumbnail.url
                        }else if( typeof attachment.sizes.thumbnail !== 'undefined' ) {
                            attachment_url = attachment.sizes.thumbnail.url
                        }else{
                            attachment_url = attachment.sizes.full.url
                        }

                        _ids.push({
                            id: attachment.id.toString(),
                        });

                        // preview
                        this.preview( attachment_url, attachment.id );

                    });

                    // set input value
                    this.$input.val( JSON.stringify( _ids ) ).trigger('input');

                    this.collect()

                }
            });

            // open the uploader dialog
            ulz_media_uploader.open();

        });

    }

    init_front() {

        this.is_multiple = this.$.data('multiple')
        this.upload_type = this.$.data('upload-type')
        this.$input = $('.ulz-upload-input', this.$)
        this.$preview = $('.ulz-image-preview', this.$)

        let gallery = []

        if( this.is_multiple ) {
            this.reorder()
        }

        this.$.on('click', '.ulz-image-remove', (e) => this.remove(e))

        $('.ulz-upload-file', this.$).on('change', (e) => {

            this.ajaxing()

            let file = e.target.files
            let files_queued = file.length
            let files_uploaded = 0
            let has_gallery_id = false
            let gallery_id = 0

            if( files_queued <= 0 ) {
                return;
            }

            if( this.is_multiple ) {
                $('.ulz-image-prv-wrapper', this.$preview).each(function() {

                    gallery_id = parseInt( $(this).attr('data-id') )

                    for( let k = 0; k < gallery.length; k++ ) {
                        if( gallery[k].id == gallery_id ) {
                            has_gallery_id = true
                        }
                    }

                    if( ! has_gallery_id ) {
                        gallery.push({
                            id: gallery_id
                        })
                    }

                })
            }

            $.each( file, ( key, value ) => {

                var data = new FormData()
                data.append( 'action', 'utillz-field-upload' )
                data.append( 'upload_type', this.upload_type )
                data.append( 'ulz_file_upload', value )
                data.append( 'ulz_action', 'upload' )

                $.ajax({
                    type: 'POST',
                    url: window.utillz_core_vars.admin_ajax,
                    data: data,
                    cache: false,
                    dataType: 'json',
                    processData: false, // don't process the files
                    contentType: false, // set content type to false as jquery will tell the server its a query string request
                    beforeSend: () => {
                        $('.ulz-error-output', this.$).empty()
                    },
                    complete: () => {
                        $('.ulz-upload-file', this.$).val('')
                    },
                    success: ( response ) => {

                        files_uploaded++

                        if( files_uploaded == files_queued ) {
                            this.$.removeClass('ulz-ajaxing')
                        }

                        if( response.success == true ) {

                            // multiple
                            if( this.is_multiple ) {

                                gallery.push({
                                    id: response.id,
                                })

                                this.$input
                                    .val( JSON.stringify( gallery ) )
                                    .trigger('input')

                            }
                            // single
                            else{

                                this.$preview.empty()
                                this.$input
                                    .val( JSON.stringify([{
                                        id: response.id,
                                    }]))
                                    .trigger('input')

                            }

                            var template = '<div class="ulz-image-prv-wrapper" data-id="' + response.id + '" data-thumb="' + response.thumb + '">' +
                                '<div class="ulz-image-prv">' +
                                    '<div class="ulz-image-prv-outer">' +
                                        '<div class="ulz-image-prv-inner ulz-flex ulz-flex-column ulz-justify-center">' +
                                            ( this.upload_type == 'image' ? '<img class="ulz-no-drag" src="' + response.thumb + '" alt="">' : '<i class="fas fa-file-alt"></i>' ) +
                                            '<span class="ulz-image-remove ulz-transition">' +
                                                '<i class="fas fa-times"></i>' +
                                            '</span>' +
                                        '</div>' +
                                    '</div>' +
                                '</div>' +
                                '<p class="ulz-file-name">' +
                                    '<a class="ulz-ellipsis" href="' + response.url + '" target="_blank">' + response.name + '</a>' +
                                '</p>' +
                            '</div>'

                            this.$preview.append( template )

                            this.collect()

                        }else{

                            $('.ulz-error-output', this.$).append('<div class="ulz-error-holder"><p class="ulz-error">' + response.error + '</p></div>')

                        }
                    }
                })
            })
        })
    }

    reorder() {

        if( this.$.data('disabled') == 'yes' ) {
            return;
        }

        //if( $.fn.sortable ) {
            $('.ulz-image-preview', this.$).sortable({
                distance: 5,
                placeholder: 'ulz-image-prv-dummy',
                tolerance: 'pointer',
                start: ( e, ui ) => {
                    ui.placeholder.height( ui.item.width() );
                },
                update: ( e, ui ) => {

                    var gallery = [];
                    var input = ui.item.closest('[data-multiple]').find('.ulz-upload-input');

                    ui.item.closest('.ulz-image-preview').find('.ulz-image-prv-wrapper').each(function() {
                        gallery.push({
                            id: $(this).attr('data-id'),
                            // thumb: $(this).attr('data-thumb'),
                        });
                    });

                    // set input value
                    input.val( JSON.stringify( gallery ) ).trigger('input');

                    this.collect()

                }
            });
        //}
    }

    preview( image_url, image_id, multiple ) {

        let preview_template = `<div class="ulz-image-prv-wrapper" data-id="${image_id}">
            <div class="ulz-image-prv">
                <div class="ulz-image-prv-outer">
                    <div class="ulz-image-prv-inner ulz-flex ulz-flex-column ulz-justify-center">` +
                        ( ( this.upload_type == 'image' && image_url !== null ) ? `<img class="ulz-no-drag" src="${image_url}" alt="">` : '<i class="fas fa-file-alt"></i>' ) +
                        `<span class="ulz-image-remove ulz-transition">
                            <i class="fas fa-times"></i>
                        </span>
                    </div>
                </div>
            </div>
        </div>`;

        if( ! this.is_multiple ) {
            this.$preview.empty();
        }

        this.$preview.append( preview_template );

    }

    remove(e) {

        // single
        if( ! this.is_multiple ) {

            // set input value
            this.$input.val('').trigger('input');
            this.$preview.empty();

            if( ! window.utillz_core_vars.is_admin ) {
                let phisical_input = $('.ulz-upload-file', this.$)
                if( phisical_input.length ) {
                    phisical_input.val('')
                }
            }

        }
        // multiple
        else{

            $( e.currentTarget ).closest('.ulz-image-prv-wrapper').remove();

            var ids = [];

            $('.ulz-image-prv-wrapper', this.$).each(function() {
                ids.push({
                    id: $(this).data('id'),
                    // thumb: $(this).data('thumb'),
                });
            });

            // set input value
            this.$input.val( JSON.stringify( ids ) ).trigger('input');

        }

        this.collect()

    }

}

/*
 * term
 *
 */
export class term extends base {

    init() {

        this.taxonomy = this.$.attr('data-taxonomy')
        this.$taxonomy = $('.ulz-terms-taxonomy select', this.$)
        this.$terms = $('.ulz-terms-terms select', this.$)
        this.$input = $('.ulz-terms-input', this.$)

        // form stack
        this.form = this.$.parent('.ulz-repeater-content')
		if( ! this.form.length ) {
			this.form = this.$.closest('.ulz-form')
		}

        this.$taxonomy.on('change', () => this.taxonomy_change())
        this.$terms.on('change', () => this.terms_change())

    }

    taxonomy_change() {

		$.ajax({
			type: 'POST',
			url: window.utillz_core_vars.admin_ajax,
			data: {
                action: 'utillz-field-term',
                taxonomy: this.$taxonomy.val()
            },
			success: response => {

				if( response.success ) {

                    this.$terms.find('option' + ( ! this.$terms.prop('multiple') ? ':gt(0)' : '' ) ).remove()

                    for ( let id in response.terms ) {

                        let option = new Option( response.terms[ id ].name, response.terms[ id ].term_id, false, false )
                        this.$terms.append( option )

                    }

                    this.$terms.closest('.ulz-terms-terms')[ response.terms.length ? 'addClass' : 'removeClass' ]('ulz-active')

                    this.collect()

				}

			},
		});

    }

    terms_change() {

        this.collect()

    }

    collect() {

        if( this.$taxonomy.val() && this.$terms.val() ) {
            this.$input.val( JSON.stringify({
                taxonomy: this.$taxonomy.val(),
                term: this.$terms.val()
            }));
        }else{
            this.$input.val( null )
        }

        super.collect();

    }

}

/*
 * order list
 *
 */
export class order_list extends base {

    init() {

        $('ul', this.$).sortable({
            distance: 5,
            cancel: '.ulz-list-empty',
            placeholder: 'ulz-placeholder',
            connectWith: $('ul', this.$),
            tolerance: 'pointer',
            update: ( e, ui ) => {

                if( ui.sender ) {

                    let sender_state = ui.sender.data('state');

                    $( 'input', ui.item ).prop('disabled', sender_state == 'active' )

                    setTimeout(() => {

                        $('>[type="hidden"]', this.$).prop('disabled', $('[data-state="active"] li', this.$).length )

                        this.collect()

                    }, 20 )

                }

            }
        });

    }

}

/*
 * autosuggest
 *
 */
export class autosuggest extends base {

    init() {

        this.xhr = null
        this.first_load = true
        this.$label = $('.ulz--label', this.$)
        this.$value = $('.ulz--value', this.$)

        this.outside()
        this.focus()

        this.$label.on('input', () => this.set_default_name())
        this.$label.on('input', () => this.autosuggest())
        this.$.on('click', '.ulz--selectable', e => this.selectable(e))
        this.$.on('click', '.ulz--listing', e => this.listing(e))
        $('.ulz--clear', this.$).on('click', e => this.clear(e))

    }

    focus() {

        this.$label.on('focus', () => {
            if( this.$label.val().length > 0 ) {
                $('.ulz-autosuggest', this.$).addClass('ulz--typed')
                if( this.first_load ) {
                    this.autosuggest()
                }
            }
            this.first_load = false
        })

    }

    outside() {

		$(document).on('mousedown.ulz-outside:quick', e => {
			if ( ! this.$.is( e.target ) && this.$.has( e.target ).length === 0 ) {
				$('.ulz-autosuggest', this.$)
                    .removeClass('ulz--selected')
                    .removeClass('ulz--typed')
			}
		})

    }

    clear() {

        $('.ulz--input input', this.$).val('')
        $('.ulz-autosuggest', this.$)
            .removeClass('ulz--typed')
            .removeClass('ulz--selected')

        this.remove_dummy()
        this.set_default_name()
        this.collect()

    }

    set_default_name() {

        setTimeout(() => {
            this.$value.attr( 'name', this.$value.attr('data-name') )
        }, 5 )

    }

    autosuggest() {

        let value = this.$label.val()

        setTimeout(() => {
            this.get_autosuggest( value )
            this.$value.val( value )
            this.collect()
        }, 5 )

    }

    get_autosuggest( search_term ) {

        $('.ulz-autosuggest', this.$)[ search_term.length > 0 ? 'addClass' : 'removeClass' ]('ulz--typed')

        if( this.xhr !== null ) {
            this.xhr.abort()
        }

        $('.ulz-autosuggest', this.$).addClass('ulz-ajaxing')

        this.xhr = $.ajax({
            type: 'post',
            dataType: 'json',
            url: window.utillz_core_vars.admin_ajax,
            data: {
                action: 'utillz-field-autosuggest',
                search_form: this.$.attr('data-search-form'),
                criteria: this.$.attr('data-criteria'),
                s: search_term,
            },
            beforeSend: ( response ) => {
                $('.ulz--results ul', this.$).html('')
                $('.ulz--results', this.$).addClass('ulz-is-empty')
            },
            success: ( response ) => {

                if( typeof response.content !== 'undefined' ) {

                    $('.ulz-autosuggest', this.$).removeClass('ulz-ajaxing')

                    let $content = $( response.content )

                    $content.imagesLoaded(() => {
                        $('.ulz--results ul', this.$).html( $content )
                        $('.ulz--results', this.$)[ response.found > 0 ? 'removeClass' : 'addClass' ]('ulz-is-empty')
                    })

                }

            }
        })

    }

    remove_dummy() {
        this.$value.val('')
    }

    selectable(e) {

        let $e = $(e.currentTarget)

        this.remove_dummy()

        this.$label
            .val( $e.find('.ulz--name').text().trim() )

        this.$value
            .attr('name', $e.attr('data-taxonomy'))
            .val( $e.attr('data-value') )

        $('.ulz-autosuggest', this.$)
            .removeClass('ulz--typed')
            .addClass('ulz--selected')

        this.collect()

    }

    get_label() {
        return $('.ulz--label', this.$ ).val();
    }

}

/*
 * color
 *
 */
export class color extends base {

    init() {

        this.$input = $('input', this.$)
        this.$input.on('input', () => this.preview() )

        this.preview()

    }

    preview() {

        let value = this.$input.val()
        let placeholder = this.$input.attr('placeholder')

        $('.ulz--preview', this.$)
            .css('background-color', '' )
            .css('background-color', value.length ? value : placeholder )

        this.collect()

    }

}

/*
 * geo
 *
 */
export class geo extends base {

    init() {

        this.set_places()
        this.set_geo()
        this.sync()

        $(`[name='ulz_geo']`, this.$).on('change', e => {
            if( $(e.currentTarget).val() == '' ) {
                $(`[name='ulz_geo_n']`, this.$).val('')
                $(`[name='ulz_geo_e']`, this.$).val('')
                $(`[name='ulz_geo_s']`, this.$).val('')
                $(`[name='ulz_geo_w']`, this.$).val('')
            }
        })

        $(`[name='ulz_geo']`, this.$).on('change', e => {
            setTimeout(() => {
                this.collect()
            }, 10 )
        })

    }

    set_places() {

        let geo_args = []

        if( window.utillz_core_vars.map.geolocation_restrictions.length ) {
            geo_args.componentRestrictions = {
                country: window.utillz_core_vars.map.geolocation_restrictions,
            }
        }

        // create the search box and link it to the UI element.
        let searchBox = new google.maps.places.Autocomplete( $(`[name='ulz_geo']`, this.$).get(0), geo_args )

        searchBox.addListener( 'place_changed', () => {

            let place = searchBox.getPlace()

            if ( ! place.geometry || ! place.geometry.location ) {
				return;
			}

            if ( place.geometry ) {

                $(`[name='ulz_geo_n']`, this.$).val( place.geometry.viewport.getNorthEast().lat )
                $(`[name='ulz_geo_e']`, this.$).val( place.geometry.viewport.getNorthEast().lng )
                $(`[name='ulz_geo_s']`, this.$).val( place.geometry.viewport.getSouthWest().lat )
                $(`[name='ulz_geo_w']`, this.$).val( place.geometry.viewport.getSouthWest().lng )

            }else{

                console.log('Returned place contains no geometry')

            }

        });

    }

    set_geo() {

        $('.ulz-geo-get', this.$).on('click', () => {

            if( navigator.geolocation ) {
                navigator.geolocation.getCurrentPosition( position => {

                    if( window.Utillz_Core.explore && window.Utillz_Core.explore.map ) {

                        if( typeof this.geocoder == 'undefined' ) {
                            this.geocoder = new google.maps.Geocoder()
                        }

                        let geo_args = {
                            'location': new google.maps.LatLng( position.coords.latitude, position.coords.longitude )
                        }

                        this.geocoder.geocode( geo_args, ( results, status ) => {
                            if( status == google.maps.GeocoderStatus.OK ) {

                                $(`[name='ulz_geo']`, this.$).val( results[0].formatted_address ).trigger('input change')

                                if ( results[0].geometry ) {

                                    $(`[name='ulz_geo_n']`, this.$).val( results[0].geometry.viewport.getNorthEast().lat )
                                    $(`[name='ulz_geo_e']`, this.$).val( results[0].geometry.viewport.getNorthEast().lng )
                                    $(`[name='ulz_geo_s']`, this.$).val( results[0].geometry.viewport.getSouthWest().lat )
                                    $(`[name='ulz_geo_w']`, this.$).val( results[0].geometry.viewport.getSouthWest().lng )

                                }else{

                                    console.log('Returned place contains no geometry')

                                }

                            }else{

                                alert(`Geocode was not successful for the following reason: ${status}`)

                            }
                        })

                    }

                }, () => {

                    alert(`Error: The Geolocation service failed`)

                })

            }else{

                alert(`Error: Your browser doesn't support geolocation`)

            }

        })

    }

    sync() {

        let $sync = $('.ulz-geo-sync input')
        if( $sync.length ) {
            $sync.off('change').on('change', e => {
                if( $sync.is(':checked') ) {
                    this.idle()
                }
            })
        }

        setTimeout(() => {
            $(document).off('utillz/explore/map/geo')
            $(document).on('utillz/explore/map/geo', () => this.idle())
        }, 500 )

    }

    idle() {

        let $sync = $('.ulz-geo-sync input')
        if( $sync.length && $sync.is(':checked') ) {

            let bounds = window.Utillz_Core.explore.map.map.getBounds()
            let center = window.Utillz_Core.explore.map.map.getCenter()
            let ne = bounds.getNorthEast()
            let sw = bounds.getSouthWest()

            $(`[name='ulz_geo_n']`, this.$).val( ne.lat() )
            $(`[name='ulz_geo_e']`, this.$).val( ne.lng() )
            $(`[name='ulz_geo_s']`, this.$).val( sw.lat() )
            $(`[name='ulz_geo_w']`, this.$).val( sw.lng() )

            window.Utillz_Core.explore.filtering()

        }

    }

    get_label() {
        return $(`input[type='text']`, this.$ ).val();
    }

}
